import json
import os
from textual.containers import Container, HorizontalGroup, Horizontal, Grid
from textual.app import ComposeResult
from textual.widgets import Label, Static, Rule, Input, Button, DataTable, Select, TabPane
from textual import events
from .messagebox import MessageBox, AskYesNo
from . import network
import time
from . import config

import uuid

class DHCP_Mapping_List(TabPane):
    def __init__(self, title, *children, name = None, id = None, classes = None, disabled = False):
        super().__init__(title, *children, name=name, id=id, classes=classes, disabled=disabled)
        
        self.click_key = None
        self.first_clicked = 0
        
    def compose(self) -> ComposeResult:
        with Container(id="li_title_container"):
            yield Label("Lefoglalt IP címek", id="li_title")
        yield Rule()
        
        with Horizontal(id="li_room_container"):
            yield Label("Szoba:", id="li_centeredlabel")
            yield Select((), prompt="Bármi", id="li_rooms")
            yield Button("Frissít", id="li_load_button")
        
        yield DataTable(id="li_device_list", zebra_stripes=True, cursor_type="row")
    
    def on_show(self):
        self.reload_ip_ranges()
        self.reload_client_list()
    
    def on_select_changed(self):
        self.reload_client_list()
    
    def reload_ip_ranges(self):
        select: Select = self.query_one("#li_rooms", Select)
        
        config.load_config()
        if not config.SETTINGS:
            self.app.push_screen(
                MessageBox("Hiba",
                            "Konfiguráció nam található... \nKérlek állítsd be a klienst a 'Beállítások' oldalon.\n\n(Esetleg elfelejtetted a beállításaid  elmenteni?)", 
                            type="error"))
            return

        API = network.OPNSenseAPI(
            url=config.SETTINGS['u'], 
            auth=(config.SETTINGS["k"], config.SETTINGS["s"]),
            verify=not config.SETTINGS["d"]
        )
        
        ranges = API.dnsmasq_get_dhcp_ranges()
        
        if ranges.status_ok:
            options = []
            for entry in ranges.data:
                options.append(
                    (
                        f"{entry.tag_name} ({entry.start_address} - {entry.end_address})",
                        entry.tag_uuid
                    )
                )
            
            select.set_options(options)
        else:
            self.app.push_screen(MessageBox("Hiba", "Nem lehetett lekérdezni az IP tartományokat.\n\n"+str(ranges.data.text), "error"))
    
    def reload_client_list(self):
        table: DataTable = self.query_one("#li_device_list", DataTable)
        select: Select = self.query_one("#li_rooms", Select)
        
        
        config.load_config()
        if not config.SETTINGS:
            self.app.push_screen(
                MessageBox("Hiba", 
                            "Konfiguráció nam található... \nKérlek állítsd be a klienst a 'Beállítások' oldalon.\n\n(Esetleg elfelejtetted a beállításaid  elmenteni?)", 
                            type="error"))
            return
        
        
        resp = network.OPNSenseAPI(
            url=config.SETTINGS['u'], auth=(config.SETTINGS["k"], config.SETTINGS["s"]),
            verify=not config.SETTINGS["d"]
            ).dnsmasq_list_mappings(tags=[select.value] if select.value is not Select.BLANK else [])
        
        if resp.status_ok:
            table.clear()
            for entry in resp.data:
                table.add_row(*[entry.name, entry.ip, entry.mac, entry.uuid])
    
    def on_mount(self):
        table: DataTable = self.query_one("#li_device_list", DataTable)
    
        table.add_column("Név")
        table.add_column("IP cím")
        table.add_column("MAC cím")
        table.add_column("UUID")
        
        # for i in range(255):
        #     table.add_row(*[f"Placeholder{i}", f"127.0.0.{i+1}", "00:00:00:00:00:00", uuid.uuid4()])
        
    def on_data_table_row_selected(self, event: DataTable.RowSelected) -> None:
        rowKey = event.row_key
        if rowKey != self.click_key:
            self.click_key = rowKey
            self.first_clicked = time.time()
        else:
            if time.time() - self.first_clicked > 0.150:
                self.first_clicked = time.time()
            else:
                self.app.push_screen(
                    AskYesNo(
                        "Biztosan ki szeretnéd törölni ezt a bejegyzést?", 
                        lambda: self.do_delete(rowKey), 
                        None)
                    )
    
    def do_delete(self, key):
        API = network.OPNSenseAPI(
            url=config.SETTINGS['u'], 
            auth=(config.SETTINGS["k"], config.SETTINGS["s"]),
            verify=not config.SETTINGS["d"]
        )
        
        table: DataTable = self.query_one("#li_device_list", DataTable)
        row: list = table.get_row(key)
        uuid = row[3]
        
        config.load_config()
        if not config.SETTINGS:
            self.app.push_screen(
                MessageBox("Hiba", 
                            "Konfiguráció nam található... \nKérlek állítsd be a klienst a 'Beállítások' oldalon.\n\n(Esetleg elfelejtetted a beállításaid  elmenteni?)", 
                            type="error"))
            return
        
        resp = API.dnsmasq_delete_mapping(uuid=uuid)
        
        if resp.status_ok:
            API.dnsmasq_apply_chnages()
            self.reload_client_list()
        else:
            self.app.push_screen(
                MessageBox("Hiba (DHCP foglalások)", 
                           "Nem sikerült törölni az eszközt: "+str(resp.data), 
                           type="error"))



    def on_button_pressed(self, event: Button.Pressed):
        if event.button.id == "li_load_button":
            self.reload_client_list()