from textual.screen import ModalScreen
from textual.containers import VerticalGroup, Container, HorizontalGroup
from typing import Callable, Literal
from textual.widgets import Static, Button
from textual.app import ComposeResult

YES = "Igen"
NO = "Nem"

class MessageBox(ModalScreen):
    CSS = """
    MessageBox {
        align: center top;
    }
    
    VerticalGroup {
        width: 75%;
        border: round $error;
        padding: 1;
        height: auto;
        align: center middle;
        content-align: center middle;
    }
    
    .title {
        text-align: center;
        width: 100%;
        background: $error 75%;
    }
    
    .top-space {
        margin: 1 0 0 0;
    }
    
    """
    
    def __init__(self, title, message, type: Literal["info", "warning", "error"] = "info"):
        super().__init__()
        self.__title = title
        self.__message = message
        self.__type = type

        self._c = {
            'info': self.app.get_theme(self.app.theme).foreground,  # type: ignore
            'warning': self.app.get_theme(self.app.theme).warning,  # type: ignore
            'error': self.app.get_theme(self.app.theme).error,      # type: ignore
        } 
        
        
    def compose(self):
        with VerticalGroup() as group:
            group.styles.border = ("round", str(self._c[self.__type]))
            
            title = Static(self.__title, classes="title")
            
            if self.__type.lower() in ('info', "warning"):
                # title.styles.color = self.app.get_theme(self.app.theme).background
                title.styles.color = "#000000"

            title.styles.background = str(self._c[self.__type])
            yield title
            yield Static(self.__message, classes="top-space")
            yield Button("Close", id="close", classes="top-space")
    
    def on_button_pressed(self, event: Button.Pressed):
        if event.button.id == "close":
            self.dismiss()


class AskYesNo(ModalScreen):
    CSS = """
    AskYesNo {  
        align: center top;
    }
    
    Container {
        border: round $foreground;
        width: 51;
        height: auto;
    }
    
    #yes {
        background: $success;
        border-left: tall $success;
        border-top: tall $success;
        border-right: tall $success 40%;
        border-bottom: tall $success 40%;
    }
    
    #yes:hover {
        background: $success 60%;
        border-left: tall $success 60%;
        border-top: tall $success 60%;
        border-right: tall $success 40%;
        border-bottom: tall $success 40%;
    }
    
    #no {
        background: $error 60%;
        border-left: tall $error 60%;
        border-top: tall $error 60%;
        border-right: tall $error 40%;
        border-bottom: tall $error 40%;
    }
    
    #no:hover {
        background: $error 40%;
        border-left: tall $error 45%;
        border-top: tall $error 45%;
        border-right: tall $error 20%;
        border-bottom: tall $error 20%;
    }
    
    .label {
        padding: 1;
        color: auto 80%;
    }
    
    .right {
        align: right middle;
    }
    """

    def __init__(self, string: str, yes: Callable | None, no: Callable | None):
        super().__init__()
        self.yes = yes
        self.no = no
        self._message_to_show = string

    def compose(self) -> ComposeResult:
        yield Container(
            Static(self._message_to_show, classes="label"),
            HorizontalGroup(
                Button(YES, id="yes"),
                Button(NO, id="no"),
                classes="right"
            )
        )

    def on_button_pressed(self, event: Button.Pressed):
        if event.button.id == "yes":
            self.dismiss()
            if self.yes is not None:
                self.yes()

        elif event.button.id == "no":
            self.dismiss()
            if self.no is not None:
                self.no()