import ipaddress
from typing import Any, TypeVar, Generic
from .network_types import *
from dataclasses import dataclass
import requests
import json

TData = TypeVar("TData")

@dataclass()
class APIResponse():
    status_code: int
    text: str
    json: dict | None = None
    
    def __init__(self, status_code: int, text: str):
        self.status_code = status_code
        self.text = text
        
        try: 
            self.json = json.loads(self.text)
        except json.decoder.JSONDecodeError:
            self.json = None
    
    def is_ok(self):
        if self.status_code != 200:
            return False
        return True

@dataclass()
class APIData(Generic[TData]):
    status_ok: bool
    data: TData

def parse_keys(s: str):
    lines = [i for i in s.split("\n") if i.strip()]
    auth_key = lines[0].split("=", 1)[1].strip()
    auth_secret = lines[1].split("=", 1)[1].strip()
    return auth_key, auth_secret

def subnet_from_range_octet(start_ip, end_ip):
    start = int(ipaddress.IPv4Address(start_ip))
    end = int(ipaddress.IPv4Address(end_ip))

    xor = start ^ end
    prefix = 32 - xor.bit_length()

    # Round DOWN to nearest multiple of 8
    prefix = (prefix // 8) * 8

    network = ipaddress.IPv4Network((start, prefix), strict=False)
    return network

class OPNSenseAPI:
    def __init__(self, url: str, auth: tuple[str, str], verify: bool | str = True):
        self.url = url
        self.auth = auth
        self.verify = verify
    
    def dnsmasq_apply_chnages(self) -> APIResponse:
        resp = requests.post(
            self.url+"/api/dnsmasq/service/reconfigure",
            auth=self.auth,
            verify=self.verify,
            json={}
        )
        return APIResponse(resp.status_code, resp.text)
    
    def dnsmasq_map_address(self, name: str, ip: IPv4 | str, mac: MAC | str, tag: str = "", description: str = "") -> APIResponse:
        resp = requests.post(
            self.url+"/api/dnsmasq/settings/add_host/",
            verify=self.verify,
            auth=self.auth,
            json={
                "host": {
                    # The important stuff
                    "host": str(name),
                    "ip": str(ip),
                    "hwaddr": str(mac),

                    # We can just ignore these
                    "domain": "",
                    "local": "0",
                    "aliases": "",
                    "cnames": "",
                    "client_id": "",
                    "lease_time": "",
                    "set_tag": str(tag),
                    "ignore": "0",
                    "descr": str(description),  # Optional description
                    "comments": ""
                }
            }
        )
        
        return APIResponse(resp.status_code, resp.text)

    def dnsmasq_get_dhcp_ranges(self) -> APIData[list[DHCPRange] | APIResponse]:
        resp = requests.post(
            self.url+"/api/dnsmasq/settings/search_range",
            verify=self.verify,
            auth=self.auth,
            json = {
                "current": 1,
                "rowCount": -1,
                "sort": {}
            }
        )
        
        response = APIResponse(resp.status_code, resp.text)
        
        if response.is_ok() and response.json:
            dhcp_ranges = []
            for row in response.json["rows"]:
                dhcp_ranges.append(
                    DHCPRange(
                        uuid=row["uuid"],
                        tag_name=row["%set_tag"],
                        tag_uuid=row["set_tag"],
                        start_address=row["start_addr"],
                        end_address=row["end_addr"],
                    )
                )
            return APIData(True, dhcp_ranges)
        else:
            return APIData(False, response)
    
    def dnsmasq_list_mappings(self, tags: list[str] = []) -> APIData[list[DHCPMapping] | APIResponse]:
        resp = requests.post(
            self.url+"/api/dnsmasq/settings/search_host",
            verify=self.verify,
            auth=self.auth,
            json = {
                "current": 1,
                "rowCount": -1,
                "sort": {},
                "tags": tags,
            }
        )
        
        response = APIResponse(resp.status_code, resp.text)
        
        if response.is_ok() and response.json:
            mappings: list[DHCPMapping] = []
            for entry in response.json["rows"]:
                mappings.append(
                    DHCPMapping(
                        uuid=entry["uuid"],
                        name=entry["host"],
                        tag_uuid=entry["set_tag"],
                        ip=entry["ip"],
                        mac=entry["hwaddr"],
                        desc="null"
                    )
                )
            return APIData(True, mappings)
        else:
            return APIData(False, response)
    
    def dnsmasq_delete_mapping(self, uuid: str):
        resp = requests.post(
            self.url+"/api/dnsmasq/settings/del_host/"+uuid,
            auth=self.auth,
            json={},
            verify=self.verify,
        )
        
        response = APIResponse(resp.status_code, resp.text)
        
        if response.is_ok() and response.json:
            return APIData(True, {})
        else:
            return APIData(False, response)