from dataclasses import dataclass
from typing import Tuple, overload


@dataclass(frozen=True)
class IPv4:
    octets: Tuple[int, int, int, int]

    @overload
    def __init__(self, addr: str) -> None:
        ...

    @overload
    def __init__(self, o1: int, o2: int, o3: int, o4: int) -> None:
        ...

    def __init__(self, *args):
        if len(args) == 1 and isinstance(args[0], str):
            parts = args[0].split(".")
            if len(parts) != 4:
                raise ValueError("Invalid IPv4 string")
            octets = tuple(int(p) for p in parts)
        elif len(args) == 4:
            octets = tuple(args)
        else:
            raise TypeError("IPv4 expects a string or 4 integers")

        if any(not (0 <= o <= 255) for o in octets):
            raise ValueError("IPv4 octets must be 0–255")

        object.__setattr__(self, "octets", octets)

    def __str__(self) -> str:
        return ".".join(map(str, self.octets))

@dataclass(frozen=True)
class MAC:
    bytes: Tuple[int, int, int, int, int, int]

    @overload
    def __init__(self, addr: str) -> None:
        ...

    @overload
    def __init__(
            self,
            b1: int, b2: int, b3: int,
            b4: int, b5: int, b6: int
    ) -> None:
        ...

    def __init__(self, *args):
        if len(args) == 1 and isinstance(args[0], str):
            parts = args[0].replace("-", ":").split(":")
            if len(parts) != 6:
                raise ValueError("Invalid MAC string")
            values = tuple(int(p, 16) for p in parts)
        elif len(args) == 6:
            values = tuple(args)
        else:
            raise TypeError("MAC expects a string or 6 integers")

        if any(not (0 <= b <= 255) for b in values):
            raise ValueError("MAC bytes must be 0–255")

        object.__setattr__(self, "bytes", values)

    def __str__(self) -> str:
        return ":".join(f"{b:02x}" for b in self.bytes)

@dataclass(frozen=True)
class DHCPRange:
    uuid: int
    tag_name: str
    tag_uuid: str
    start_address: str
    end_address: str

@dataclass(frozen=True)
class DHCPMapping:
    uuid: int
    name: str
    tag_uuid: list[str]
    ip: str
    mac: str
    desc: str
