import json
import os
from textual.containers import Container, HorizontalGroup, Horizontal, VerticalScroll
from textual.app import ComposeResult
from textual.widgets import Label, Static, Rule, Input, Button, TabPane, Select
from textual import events
from .messagebox import MessageBox
from . import network
from . import config
import ipaddress

class New_DHCP_Mapping(TabPane):
    def __init__(self, title, *children, name = None, id = None, classes = None, disabled = False):
        super().__init__(title, *children, name=name, id=id, classes=classes, disabled=disabled)
        
        self.ranges = []
        
    def compose(self) -> ComposeResult:
        with Container(classes="title_container"):
            yield Label("DHCP foglalás hozzáadása", id="dhcp_title")
        yield Rule()

        with VerticalScroll():
            with Horizontal(id="dn_room_container"):
                yield Label("Tartomány:", classes="centered-label")
                yield Select((), prompt="Bármi", id="dn_room_selector")
                yield Button("Frissít", id="dn_refresh")
            
            with Container(classes="grid"):
                yield Label("Gép neve: ", classes="centered-label")
                yield Input(id="dhcp_hostname", type="text", placeholder="Gépnév123")

                yield Label("IP cím: ", classes="centered-label")
                with Horizontal(classes="grouped_input"):
                    yield Input(id="dhcp_ip_0", type="number", classes="ip_octet", placeholder="xxx")
                    yield Label(".", classes="centered-label")
                    yield Input(id="dhcp_ip_1", type="number", classes="ip_octet", placeholder="xxx")
                    yield Label(".", classes="centered-label")
                    yield Input(id="dhcp_ip_2", type="number", classes="ip_octet", placeholder="xxx")
                    yield Label(".", classes="centered-label")
                    yield Input(id="dhcp_ip_3", type="number", classes="ip_octet", placeholder="xxx")
                
                yield Label("MAC cím: ", classes="centered-label")
                with Horizontal(classes="grouped_input"):
                    yield Input(id="dhcp_mac_0", type="text", classes="mac_byte", placeholder="xx")
                    yield Label(":", classes="centered-label")
                    yield Input(id="dhcp_mac_1", type="text", classes="mac_byte", placeholder="xx")
                    yield Label(":", classes="centered-label")
                    yield Input(id="dhcp_mac_2", type="text", classes="mac_byte", placeholder="xx")
                    yield Label(":", classes="centered-label")
                    yield Input(id="dhcp_mac_3", type="text", classes="mac_byte", placeholder="xx")
                    yield Label(":", classes="centered-label")
                    yield Input(id="dhcp_mac_4", type="text", classes="mac_byte", placeholder="xx")
                    yield Label(":", classes="centered-label")
                    yield Input(id="dhcp_mac_5", type="text", classes="mac_byte", placeholder="xx")
                    
            
            yield Button("Hozzáad", id="dhcp_send")
    
    def load_ranges(self):
        select: Select = self.query_one("#dn_room_selector", Select)
        
        current = select.value
        
        config.load_config()
        if not config.SETTINGS:
            self.app.push_screen(
                MessageBox("Hiba", 
                            "Konfiguráció nam található... \nKérlek állítsd be a klienst a 'Beállítások' oldalon.\n\n(Esetleg elfelejtetted a beállításaid  elmenteni?)", 
                            type="error"))
            return

        API = network.OPNSenseAPI(
            url=config.SETTINGS['u'], 
            auth=(config.SETTINGS["k"], config.SETTINGS["s"]),
            verify=not config.SETTINGS["d"]
        )
        
        ranges = API.dnsmasq_get_dhcp_ranges()
        
        if ranges.status_ok:
            self.ranges.clear()
            self.ranges = ranges.data.copy()
            options = []
            set_after = False
            for entry in ranges.data:
                options.append(
                    (
                        f"{entry.tag_name} ({entry.start_address} - {entry.end_address})",
                        entry.tag_uuid
                    )
                )
                if entry.tag_uuid == current:
                    set_after = True
            select.set_options(options=options)
            if set_after:
                select.value = current
        else:
            self.app.push_screen(MessageBox("Hiba", "Nem lehetett lekérdezni az IP tartományokat.\n\n"+str(ranges.data.text), "error"))
        
    def on_show(self) -> None:
        self.load_ranges()
    
    def on_select_changed(self, event: Select.Changed):
        if event.select.id == "dn_room_selector":
            uuid = event.select.value
            if uuid == Select.BLANK:
                for i in range(4):
                    widget = self.query_one(f"#dhcp_ip_{i}", Input)
                    widget.value = ""
                    widget.disabled = False
                return
            
            dhcp_range = [i for i in self.ranges if i.tag_uuid == uuid][0]
            
            # self.app.push_screen(MessageBox("Debug", repr(dhcp_range)))
            # return
            
            subnet = network.subnet_from_range_octet(dhcp_range.start_address, dhcp_range.end_address)
            
            mask = subnet.netmask
            
            for i in range(4):
                if mask.packed[i] == 255:
                    widget = self.query_one(f"#dhcp_ip_{i}", Input)
                    widget.value = str(ipaddress.IPv4Address(str(dhcp_range.start_address)).packed[i])
                    widget.disabled = True
            
    
    def on_button_pressed(self, event: Button.Pressed):
        room_select: Select = self.query_one("#dn_room_selector", Select)
        tag = room_select.value if room_select.value != Select.BLANK else ""
        
        config.load_config()
        if not config.SETTINGS:
            self.app.push_screen(
                MessageBox("Hiba", 
                           "Konfiguráció nam található... \nKérlek állítsd be a klienst a 'Beállítások' oldalon.\n\n(Esetleg elfelejtetted a beállításaid  elmenteni?)", 
                           type="error"))
            return
        
        API = network.OPNSenseAPI(
                config.SETTINGS["u"],
                (config.SETTINGS["k"], config.SETTINGS["s"]),
                not config.SETTINGS.get("d"),
                )

        if event.button.id == "dhcp_send":
            config.load_config()
            
            ip_address = []
            mac_address = []
            try:
                for i in range(4):
                    ip_address.append(int(self.query_one(f"#dhcp_ip_{i}", Input).value))
            except ValueError:
                self.app.push_screen(MessageBox("Hiba", "Érvénytelen IP cím", "error"))
                return
                
            try:
                for i in range(6):
                    mac_address.append(int(self.query_one(f"#dhcp_mac_{i}", Input).value, 16))
            except ValueError:
                self.app.push_screen(MessageBox("Hiba", "Érvénytelen MAC cím", "error"))
                return
            
            resp = API.dnsmasq_map_address(
                    name=self.query_one("#dhcp_hostname", Input).value,
                    ip=network.IPv4(*ip_address),
                    mac=network.MAC(*mac_address),
                    tag=tag,
                )
            
            if resp.is_ok() and resp.json.get("result") == "saved":
                API.dnsmasq_apply_chnages()
                self.app.push_screen(MessageBox("Siker", f"DHCP kiosztás sikeresen hozzáadva."))
            else:
                self.app.push_screen(MessageBox("Hiba", f"A művelet sikertelen volt: \n{resp.text}"))
        elif event.button.id == "dn_refresh":
            self.load_ranges()
            